#ifndef AHLGREN_BENCHMARK
#define AHLGREN_BENCHMARK

#include <iostream>
#include <vector>
#include <numeric>
#include <algorithm>
#include <array>
#include <cmath>
#include <valarray>

#include "global.h"
#include "program.h"

namespace lp {

	typedef std::multimap<sign_t,lp::stat> stats_tbl;
	typedef std::vector<std::pair<sign_t,lp::stat>> stat_vec;

	// Print Statistics
	void print_statistics(const stats_tbl&);
	template <typename Iter> void print_statistics(Iter beg, Iter end);

	// Run Kfold Cross Validation
	struct validation_algorithm {
		validation_algorithm(std::string m, parameters s) : method(std::move(m)), settings(std::move(s)), result() {}
		std::string method;
		parameters settings;
		stats_tbl result;
	};

	// Recursively validate directories
	stats_tbl validate_rec(const std::string& method, Program kb, const boost::filesystem::path& folder);
	// Validate this path
	stats_tbl validate(const std::string& method, Program kb, const boost::filesystem::path& folder);

	// Run a benchmark of count trials (no validation)
	lp::Program benchmark(int count, const char* method, const lp::Program& kb);

	// Automated cross validation without files
	void auto_cross_validation(int count, int kfold, const std::string& method_mask, const lp::Program& kb);
	void auto_cross_validation(int kfold, std::vector<validation_algorithm>& methods, lp::Program kb);


	//===================================== Definitions =====================================//

	template <typename Iter>
	void print_statistics(Iter beg, Iter end)
	{
		const auto size = std::distance(beg,end);
		cout << setw(25) << std::left << "Sample Size:" << setw(10) << size << "\n";
		if (size == 0) return;
		const auto old_prec = cout.precision(3);
		cout << std::left << std::fixed;
		const lp::stat avg = average(beg,end);
		const lp::stat dev = stddev(beg,end);
		const lp::stat min = lp::stat_min(beg,end);
		const lp::stat max = lp::stat_max(beg,end);
		//cout << setfill('-') << setw(25+4*10) << setfill(' ') << "\n";
		cout << setw(25) << " <Value>" << setw(10) << "Mean" << setw(10) << "Low" << setw(10) << "High" << setw(10) <<  "Dev" << "\n";
		cout << setw(25) << "Execution Time" << setw(10) << avg[stat::time]/1000.0 << setw(10) << min[stat::time]/1000.0 << setw(10) << max[stat::time]/1000.0 << setw(10) << dev[stat::time]/1000.0 << "\n";
		cout << setw(25) << "Fitness Evaluations" << setw(10) << avg[stat::fitc] << setw(10) << min[stat::fitc] << setw(10) << max[stat::fitc] << setw(10) << dev[stat::fitc] << "\n";
		cout << setw(25) << "Examples Evaluated" << setw(10) << avg[stat::exc] << setw(10) << min[stat::exc] << setw(10) << max[stat::exc] << setw(10) << dev[stat::exc] << "\n";
		cout << setw(25) << "Solution Size" << setw(10) << avg[stat::complexity] << setw(10) << min[stat::complexity] << setw(10) << max[stat::complexity] << setw(10) << dev[stat::complexity] << "\n";
		cout << setw(25) << "Clauses Added" << setw(10) << avg[stat::cl_added] << setw(10) << min[stat::cl_added] << setw(10) << max[stat::cl_added] << setw(10) << dev[stat::cl_added] << "\n";
		cout << setw(25) << "Bottom Constructed" << setw(10) << avg[stat::bottomc] << setw(10) << min[stat::bottomc] << setw(10) << max[stat::bottomc] << setw(10) << dev[stat::bottomc] << "\n";
		cout << setw(25) << "Searches Exhausted" << setw(10) << avg[stat::exhausted] << setw(10) << min[stat::exhausted] << setw(10) << max[stat::exhausted] << setw(10) << dev[stat::exhausted] << "\n";
		cout << setw(25) << "Searches Aborted" << setw(10) << avg[stat::aborted] << setw(10) << min[stat::aborted] << setw(10) << max[stat::aborted] << setw(10) << dev[stat::aborted] << "\n";
		cout << setw(25) << "Inductions Cancelled" << setw(10) << avg[stat::full_abort] << setw(10) << min[stat::full_abort] << setw(10) << max[stat::full_abort] << setw(10) << dev[stat::full_abort] << "\n";
		// Valid / Consistent / Inconsistent
		cout << setw(25) << "Consistent" << setw(10) << avg[stat::conc] << setw(10) << min[stat::conc] << setw(10) << max[stat::conc] << setw(10) << dev[stat::conc] << "\n";
		cout << setw(25) << "Inconsistent" << setw(10) << avg[stat::inconc] << setw(10) << min[stat::inconc] << setw(10) << max[stat::inconc] << setw(10) << dev[stat::inconc] << "\n";
		cout << setw(25) << "Invalid" << setw(10) << avg[stat::invalid] << setw(10) << min[stat::invalid] << setw(10) << max[stat::invalid] << setw(10) << dev[stat::invalid] << "\n";
		// Compute Proportion Valid
		const double pvalid = (avg[stat::conc] + avg[stat::inconc])/(avg[stat::conc]+avg[stat::inconc]+avg[stat::invalid]);
		const double pvaliddev = std::sqrt(pvalid*(1-pvalid)/std::distance(beg,end));
		cout << setw(25) << "Proportion Valid" << setw(10) << pvalid << setw(10) << " " << setw(10) << " " << setw(10) << pvaliddev << "\n";
		// Compute Proportion Consistent
		const double pcon = avg[stat::conc]/(avg[stat::conc]+avg[stat::inconc]);
		const double pdev = std::sqrt(pcon*(1-pcon)/std::distance(beg,end));
		cout << setw(25) << "Proportion Consistent" << setw(10) << pcon << setw(10) << " " << setw(10) << " " << setw(10) << pdev << "\n";				
		const double pfull = std::count_if(beg,end,[](const lp::stat& l){return l[stat::exhausted] == l[stat::bottomc];}) / double(size);
		cout << setw(25) << "Proportion Completed" << setw(10) << pfull << "\n";
		//// Compute Precision
		//auto prec_ratio = [](const lp::stat& s) -> double { return double(s[stat::recall_pass]) / s[stat::recall_size]; };
		//const double recall = avg[stat::recall_pass] / avg[stat::recall_size];
		//const double recalldev = std::sqrt(recall*(1.0-recall) / size);
		//auto minmax_it = std::minmax_element(beg,end,[&](const stat& s, const stat& t){ return prec_ratio(s) < prec_ratio(t); });
		//auto minmax = std::make_pair( prec_ratio(*minmax_it.first), prec_ratio(*minmax_it.second) );
		//cout << setw(25) << "Average Precision" << setw(10) << recall << setw(10) << minmax.first << setw(10) << minmax.second << setw(10) << recalldev << "\n";
		// Compute Accuracy
		// Accuracy == (True pos + True neg) / Everything
		double tp = 0, fp = 0, tn = 0, fn = 0, acc_min = 1, acc_max = 0;
		std::for_each(beg,end,[&](const stat& s){
			tp += s[stat::true_positive];
			fp += s[stat::false_positive];
			tn += s[stat::true_negative];
			fn += s[stat::false_negative];
			const double hits = s[stat::true_positive] + s[stat::true_negative];
			const double tot = s[stat::true_positive] + s[stat::true_negative] + s[stat::false_positive] + s[stat::false_negative];
			const double this_acc = hits/tot;
			if (this_acc < acc_min) acc_min = this_acc;
			if (this_acc > acc_max) acc_max = this_acc;
		});
		cout << setw(25) << "True Positives" << setw(10) << tp << "\n";
		cout << setw(25) << "True Negatives" << setw(10) << tn << "\n";
		cout << setw(25) << "False Positives" << setw(10) << fp << "\n";
		cout << setw(25) << "False Negatives" << setw(10) << fn << "\n";
		const double accuracy = (tp+tn) / (tp+tn+fp+fn);
		cout << setw(25) << "Accuracy" << setw(10) << accuracy << setw(10) << acc_min << setw(10) << acc_max << setw(10) << "-" << "\n";

		// reset precision
		cout.precision(old_prec);
	}


	//template <typename Titer, typename Viter>
	//std::vector<std::pair<sign_t,lp::stat>> validate(
	//	const std::string& method,
	//	Program kb,
	//	Titer tbeg, 
	//	Titer tend,
	//	Viter vbeg,
	//	Viter vend) 
	//{
	//	std::vector<std::pair<sign_t,lp::stat>> statsv;
	//	// Put all training examples in kb
	//	std::for_each(tbeg,tend,[&](const Functor& cl){ kb.push_back(cl); });
	//	// Induce
	//	auto stats = kb.generalize(method.c_str());
	//	// Map signatures to stats vector index
	//	std::map<sign_t,int> sign2index;
	//	for (decltype(stats.size()) k = 0; k < stats.size(); ++k) {
	//		auto p = sign2index.insert(std::make_pair(stats[k].first,k));
	//		assert(p.second);
	//	}
	//	// Check coverage
	//	std::for_each(vbeg,vend,[&](const Functor& cl){
	//		if (kb.covers(cl)) {
	//			const int index = sign2index.find(cl.signature())->second;
	//			++(stats[index].second[stat::recall_pass]);
	//		}
	//	});
	//	return stats;
	//}


}


#endif



